/*
 * Decompiled with CFR 0.152.
 */
package fr.xephi.authme.service;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.maxmind.db.CHMCache;
import com.maxmind.db.NodeCache;
import com.maxmind.db.Reader;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.exception.GeoIp2Exception;
import com.maxmind.geoip2.model.AbstractCountryResponse;
import com.maxmind.geoip2.record.AbstractNamedRecord;
import com.maxmind.geoip2.record.Country;
import fr.xephi.authme.ConsoleLogger;
import fr.xephi.authme.initialization.DataFolder;
import fr.xephi.authme.output.ConsoleLoggerFactory;
import fr.xephi.authme.service.BukkitService;
import fr.xephi.authme.settings.Settings;
import fr.xephi.authme.settings.properties.ProtectionSettings;
import fr.xephi.authme.util.FileUtils;
import fr.xephi.authme.util.InternetProtocolUtils;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.Objects;
import java.util.Optional;
import java.util.zip.GZIPInputStream;
import javax.inject.Inject;

public class GeoIpService {
    private static final String LICENSE = "[LICENSE] This product includes GeoLite2 data created by MaxMind, available at https://www.maxmind.com";
    private static final String DATABASE_NAME = "GeoLite2-Country";
    private static final String DATABASE_FILE = "GeoLite2-Country.mmdb";
    private static final String DATABASE_TMP_FILE = "GeoLite2-Country.mmdb.tmp";
    private static final String ARCHIVE_FILE = "GeoLite2-Country.mmdb.gz";
    private static final String ARCHIVE_URL = "https://updates.maxmind.com/geoip/databases/GeoLite2-Country/update";
    private static final int UPDATE_INTERVAL_DAYS = 30;
    private final ConsoleLogger logger = ConsoleLoggerFactory.get(GeoIpService.class);
    private final Path dataFile;
    private final BukkitService bukkitService;
    private final Settings settings;
    private DatabaseReader databaseReader;
    private volatile boolean downloading;

    @Inject
    public GeoIpService(@DataFolder File dataFolder, BukkitService bukkitService, Settings settings) {
        this.bukkitService = bukkitService;
        this.dataFile = dataFolder.toPath().resolve(DATABASE_FILE);
        this.settings = settings;
        this.isDataAvailable();
    }

    @VisibleForTesting
    GeoIpService(@DataFolder File dataFolder, BukkitService bukkitService, Settings settings, DatabaseReader reader) {
        this.bukkitService = bukkitService;
        this.settings = settings;
        this.dataFile = dataFolder.toPath().resolve(DATABASE_FILE);
        this.databaseReader = reader;
    }

    private synchronized boolean isDataAvailable() {
        if (!((Boolean)this.settings.getProperty(ProtectionSettings.ENABLE_GEOIP)).booleanValue()) {
            return false;
        }
        if (this.downloading) {
            return false;
        }
        if (this.databaseReader != null) {
            return true;
        }
        if (Files.exists(this.dataFile, new LinkOption[0])) {
            try {
                FileTime lastModifiedTime = Files.getLastModifiedTime(this.dataFile, new LinkOption[0]);
                if (Duration.between(lastModifiedTime.toInstant(), Instant.now()).toDays() <= 30L) {
                    this.startReading();
                    return true;
                }
                this.logger.debug("GEO IP database is older than 30 Days");
            }
            catch (IOException ioEx) {
                this.logger.logException("Failed to load GeoLiteAPI database", ioEx);
                return false;
            }
        }
        this.downloading = true;
        this.bukkitService.runTaskAsynchronously(this::updateDatabase);
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateDatabase() {
        this.logger.info("Downloading GEO IP database, because the old database is older than 30 days or doesn't exist");
        Path downloadFile = null;
        Path tempFile = null;
        try {
            downloadFile = Files.createTempFile(ARCHIVE_FILE, null, new FileAttribute[0]);
            tempFile = Files.createTempFile(DATABASE_TMP_FILE, null, new FileAttribute[0]);
            String expectedChecksum = this.downloadDatabaseArchive(downloadFile);
            if (expectedChecksum == null) {
                this.logger.info("There is no newer GEO IP database uploaded to MaxMind. Using the old one for now.");
                this.startReading();
                return;
            }
            this.extractDatabase(downloadFile, tempFile);
            this.verifyChecksum(Hashing.md5(), tempFile, expectedChecksum);
            Files.copy(tempFile, this.dataFile, StandardCopyOption.REPLACE_EXISTING);
            this.logger.info("Successfully downloaded new GEO IP database to " + String.valueOf(this.dataFile));
            this.startReading();
        }
        catch (IOException ioEx) {
            this.logger.logException("Could not download GeoLiteAPI database", ioEx);
        }
        finally {
            if (downloadFile != null) {
                FileUtils.delete(downloadFile.toFile());
            }
            if (tempFile != null) {
                FileUtils.delete(tempFile.toFile());
            }
        }
    }

    private void startReading() throws IOException {
        this.databaseReader = new DatabaseReader.Builder(this.dataFile.toFile()).withCache((NodeCache)new CHMCache()).fileMode(Reader.FileMode.MEMORY).build();
        this.logger.info(LICENSE);
        this.downloading = false;
    }

    private String downloadDatabaseArchive(Instant lastModified, Path destination) throws IOException {
        String clientId = (String)this.settings.getProperty(ProtectionSettings.MAXMIND_API_CLIENT_ID);
        String licenseKey = (String)this.settings.getProperty(ProtectionSettings.MAXMIND_API_LICENSE_KEY);
        if (clientId.isEmpty() || licenseKey.isEmpty()) {
            this.logger.warning("No MaxMind credentials found in the configuration file! GeoIp protections will be disabled.");
            return null;
        }
        HttpURLConnection connection = (HttpURLConnection)new URL(ARCHIVE_URL).openConnection();
        String basicAuth = "Basic " + new String(Base64.getEncoder().encode((clientId + ":" + licenseKey).getBytes()));
        connection.setRequestProperty("Authorization", basicAuth);
        if (lastModified != null) {
            ZonedDateTime zonedTime = lastModified.atZone(ZoneId.of("GMT"));
            String timeFormat = DateTimeFormatter.RFC_1123_DATE_TIME.format(zonedTime);
            connection.addRequestProperty("If-Modified-Since", timeFormat);
        }
        if (connection.getResponseCode() == 304) {
            connection.getInputStream().close();
            return null;
        }
        String hash = connection.getHeaderField("X-Database-MD5");
        String rawModifiedDate = connection.getHeaderField("Last-Modified");
        Instant modifiedDate = Instant.from(DateTimeFormatter.RFC_1123_DATE_TIME.parse(rawModifiedDate));
        Files.copy(connection.getInputStream(), destination, StandardCopyOption.REPLACE_EXISTING);
        Files.setLastModifiedTime(destination, FileTime.from(modifiedDate));
        return hash;
    }

    private String downloadDatabaseArchive(Path destination) throws IOException {
        Instant lastModified = null;
        if (Files.exists(this.dataFile, new LinkOption[0])) {
            lastModified = Files.getLastModifiedTime(this.dataFile, new LinkOption[0]).toInstant();
        }
        return this.downloadDatabaseArchive(lastModified, destination);
    }

    private void verifyChecksum(HashFunction function, Path file, String expectedChecksum) throws IOException {
        HashCode expectedHash;
        HashCode actualHash = function.hashBytes(Files.readAllBytes(file));
        if (!Objects.equals(actualHash, expectedHash = HashCode.fromString((String)expectedChecksum))) {
            throw new IOException("GEO IP Checksum verification failed. Expected: " + expectedChecksum + "Actual:" + String.valueOf(actualHash));
        }
    }

    private void extractDatabase(Path inputFile, Path outputFile) throws IOException {
        try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(inputFile, new OpenOption[0]));
             GZIPInputStream gzipIn = new GZIPInputStream(in);){
            Files.copy(gzipIn, outputFile, StandardCopyOption.REPLACE_EXISTING);
            Files.setLastModifiedTime(outputFile, Files.getLastModifiedTime(inputFile, new LinkOption[0]));
        }
    }

    public String getCountryCode(String ip) {
        if (InternetProtocolUtils.isLocalAddress(ip)) {
            return "LOCALHOST";
        }
        return this.getCountry(ip).map(Country::getIsoCode).orElse("--");
    }

    public String getCountryName(String ip) {
        if (InternetProtocolUtils.isLocalAddress(ip)) {
            return "LocalHost";
        }
        return this.getCountry(ip).map(AbstractNamedRecord::getName).orElse("N/A");
    }

    private Optional<Country> getCountry(String ip) {
        if (ip == null || ip.isEmpty() || !this.isDataAvailable()) {
            return Optional.empty();
        }
        try {
            InetAddress address = InetAddress.getByName(ip);
            return Optional.ofNullable(this.databaseReader.country(address)).map(AbstractCountryResponse::getCountry);
        }
        catch (UnknownHostException address) {
        }
        catch (GeoIp2Exception | IOException ioEx) {
            this.logger.logException("Cannot lookup country for " + ip + " at GEO IP database", ioEx);
        }
        return Optional.empty();
    }
}

